unit XMLBlocks;

{
  XML Building Blocks
  Plug-and-play components for working with XML expressed as DOM documents.
  Producers generate new XML documents as DOMs.
  Consumers use those DOMs to amend or process them.

  Included are:
  TXBBParser          - generate DOMs from existing documents
  TXBBSQL             - generates DOMs from database queries
  TXBBTextFile        - generates DOMs around a text file's contents
  TXBBComponent       - generates DOMs from Delphi component properties

  TXBBMerge           - combines several DOMs into one new DOM
  TXBBFork            - sends a DOM to several consumers

  TXBBTimestamp       - add a timestamp element to a DOM
  TXBBTransform       - apply an XSL transformation to a DOM

  TXBBWriter          - send a DOM to a file or stream
  TXBBTreeView        - display a DOM in a tree view
  TXBBMemo            - display a DOM in a memo
  TXBBStringGrid      - display a DOM in a string grid
  TXBBWebBrowser      - display a DOM in a Web browser
  TXBBComponentCreate - create a component from the DOM

  Written by Keith Wood (kbwood@iprimus.com.au)
  Version 1.0 - 7 March 2002.
}

interface

uses
  Classes, SysUtils, StrUtils, Math, Graphics, Controls, ComCtrls, StdCtrls,
  ImgList, Grids, DB, DBTables, ShDocVw, ActiveX, XMLDOM;

type
  { Definition for the user of an XML document in DOM format }
  IXMLConsumer = interface
    ['{917863FF-96D1-40F9-9868-926D9C299068}']
    { Pass the DOM document along to be used and start processing it }
    procedure DocumentReady(Document: IDOMDocument);
  end;

  { Definition for the generator of an XML document in DOM format }
  IXMLProducer = interface
    ['{B53AF472-4B85-4F98-98B4-48501C81AE6A}']
    { Generate a new XML document and pass it to the consumer }
    procedure CreateDocument;
    function GetConsumer: IXMLConsumer;
    procedure SetConsumer(const Value: IXMLConsumer);
    { The consumer makes use of the new document }
    property Consumer: IXMLConsumer read GetConsumer write SetConsumer;
  end;

  { An XML Building Blocks specific exception }
  EXBBException = class(Exception)
  end;

  { Event to modify a newly created element }
  TXBBTagCreateEvent = procedure (Sender: TObject; Element: IDOMElement)
    of object;

  { Base class for developing consumers and producers.
    It's producer part (CreateDocument) does nothing. Override this for
    a new producer to actually generate the document and call DocumentReady.
    It's consumer part (DocumentReady) calls ProcessDocument to work on
    the document, then passes it along to any attached consumer.
    ProcessDocument just returns the original document without change.
    Override this for a new consumer. }
  TXMLBuildingBlock = class(TComponent, IXMLConsumer, IXMLProducer)
  private
    FConsumer: IXMLConsumer;
    FOnTagCreate: TXBBTagCreateEvent;
    FTagName: string;
    function GetConsumer: IXMLConsumer;
    procedure SetConsumer(const Value: IXMLConsumer);
  protected
    property Consumer: IXMLConsumer read GetConsumer write SetConsumer;
    property TagName: string read FTagName write FTagName;
    property OnTagCreate: TXBBTagCreateEvent read FOnTagCreate
      write FOnTagCreate;
    procedure DoOnTagCreate(Element: IDOMElement);
    function NewDocument(const TagName: string): IDOMDocument; virtual;
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    procedure NotifyConsumer(Document: IDOMDocument);
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument; virtual;
  public
    procedure CreateDocument; virtual;
    procedure DocumentReady(Document: IDOMDocument); virtual;
  end;

  { Create a new DOM from an existing XML document.
    The document can come from an existing file (XMLSource),
    from a stream (XMLStream), or from memory (XMLText). }
  TXBBParser = class(TXMLBuildingBlock)
  private
    FXMLSource: TFileName;
    FXMLStream: TStream;
    FXMLText: TStrings;
  public
    constructor Create(AOwner: TComponent); overload; override;
    constructor Create(AOwner: TComponent; const XMLSource: TFileName);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const XMLStream: TStream);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const XMLText: TStrings);
      reintroduce; overload;
    destructor Destroy; override;
    procedure CreateDocument; override;
  published
    property Consumer;
    property XMLSource: TFileName read FXMLSource write FXMLSource;
    property XMLStream: TStream read FXMLStream write FXMLStream;
    property XMLText: TStrings read FXMLText write FXMLText;
  end;

  { Event to modify a newly created record element }
  TXBBRecordTagCreateEvent = procedure (Sender: TObject; Element: IDOMElement;
    Dataset: TDataSet) of object;

  TXBBFieldFormat = (xfText, xfElement, xfAttributeOnly);

  { Create a new DOM from an SQL query.
    The DatabaseName and SQL properties retrieve data which is then made
    into an XML document. The top-level tag is named according to the
    TagName property, or from the DatabaseName if this is blank.
    Each row from the query becomes an element under this, using
    RecordTagName as its name, or 'record' if this is blank.
    Fields from the query then become child elements of the record,
    with names taken from the field name, and contents as text or
    CDATA section (if they contain '<' or '>') nodes. }
  TXBBSQL = class(TXMLBuildingBlock)
  private
    FDatabaseName: string;
    FFieldFormat: TXBBFieldFormat;
    FOnRecordTagCreate: TXBBRecordTagCreateEvent;
    FRecordTagName: string;
    FSQL: TStrings;
    procedure GetText(Sender: TField; var Text: String; DisplayText: Boolean);
  public
    constructor Create(AOwner: TComponent); overload; override;
    constructor Create(AOwner: TComponent; const TagName: string);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const DatabaseName: string;
      const SQL: TStrings; const TagName: string = ''); reintroduce; overload;
    destructor Destroy; override;
    procedure CreateDocument; override;
  published
    property Consumer;
    property DatabaseName: string read FDatabaseName write FDatabaseName;
    property FieldFormat: TXBBFieldFormat read FFieldFormat write FFieldFormat
      default xfText;
    property RecordTagName: string read FRecordTagName write FRecordTagName;
    property SQL: TStrings read FSQL write FSQL;
    property TagName;
    property OnRecordTagCreate: TXBBRecordTagCreateEvent read FOnRecordTagCreate
      write FOnRecordTagCreate;
    property OnTagCreate;
  end;

  { Create a new DOM around the contents of a text file.
    The FileName property locates the file which is then made
    into an XML document. The top-level tag is named according to the
    TagName property, or 'file' if this is blank. It contains a single
    text or CDATA section node (depending on the AsCDATA property)
    that has the file's contents. }
  TXBBTextFile = class(TXMLBuildingBlock)
  private
    FAsCDATA: Boolean;
    FFileName: TFileName;
  public
    constructor Create(AOwner: TComponent; const FileName: TFileName = '';
      const TagName: string = ''); reintroduce; overload;
    procedure CreateDocument; override;
  published
    property AsCDATA: Boolean read FAsCDATA write FAsCDATA default False;
    property Consumer;
    property FileName: TFileName read FFileName write FFileName;
    property TagName;
    property OnTagCreate;
  end;

  { Create a new DOM around the properties of a component.
    Components appear within 'object' elements with 'name' and 'type'
    attributes. Properties appear within embedded 'property' elements
    with 'name' and 'value' attributes. Collections items appear as
    'item' elements within a property. }
  TXBBComponent = class(TXMLBuildingBlock)
  private
    FComponent: TComponent;
  public
    constructor Create(AOwner: TComponent; const TagName: string = '';
      const Component: TComponent = nil); reintroduce; overload;
    procedure CreateDocument; override;
  published
    property Component: TComponent read FComponent write FComponent;
    property Consumer;
    property TagName;
    property OnTagCreate;
  end;

  { Write out the DOM to a given file or stream.
    Set one of FileName or Stream before using this component. }
  TXBBWriter = class(TXMLBuildingBlock)
  private
    FFileName: TFileName;
    FStream: TStream;
  protected
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent; const FileName: TFileName);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const Stream: TStream);
      reintroduce; overload;
  published
    property Consumer;
    property FileName: TFileName read FFileName write FFileName;
    property Stream: TStream read FStream write FStream;
  end;

  { Merge several DOM documents together in a new document under a new
    main element (named from the TagName property).
    Set the number of documents to expect with the NumDocuments property.
    When this many have appeared via the DocumentReady method and been
    combined, they are sent on to this component's consumer.
    Reset the component for another combination by setting NumDocuments again. }
  TXBBMerge = class(TXMLBuildingBlock)
  private
    FCountDown: Integer;
    FMergedDocument: IDOMDocument;
    FNumDocuments: Integer;
    procedure SetNumDocuments(Value: Integer);
  public
    constructor Create(AOwner: TComponent; const NumDocuments: Integer = 2;
      const TagName: string = ''); reintroduce; overload;
    procedure DocumentReady(Document: IDOMDocument); override;
  published
    property Consumer;
    property NumDocuments: Integer read FNumDocuments write SetNumDocuments;
    property TagName;
    property OnTagCreate;
  end;

  { A consumer item for a collection. }
  TXBBConsumerCollectionItem = class(TCollectionItem)
  private
    FConsumer: IXMLConsumer;
    FName: string;
    procedure SetConsumer(Value: IXMLConsumer);
    procedure SetName(Value: string);
  protected
    function GetDisplayName: string; override;
  published
    property Consumer: IXMLConsumer read FConsumer write SetConsumer;
    property Name: string read FName write SetName;
  end;

  { A collection of consumers. }
  TXBBConsumerCollection = class(TCollection)
  private
    FOwner: TPersistent;
  protected
    function GetItem(Index: Integer): TXBBConsumerCollectionItem;
    function  GetOwner: TPersistent; override;
    procedure SetItem(Index: Integer; Value: TXBBConsumerCollectionItem);
  public
    constructor Create(Owner: TPersistent);
    function Add: TXBBConsumerCollectionItem;
    function FindItemID(ID: Integer): TXBBConsumerCollectionItem;
    function Insert(Index: Integer): TXBBConsumerCollectionItem;
    property Items[Index: Integer]: TXBBConsumerCollectionItem
      read GetItem write SetItem;
  end;

  { Pass a DOM document off to several consumers. }
  TXBBFork = class(TXMLBuildingBlock)
  private
    FConsumers: TXBBConsumerCollection;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DocumentReady(Document: IDOMDocument); override;
  published
    property Consumers: TXBBConsumerCollection read FConsumers write FConsumers;
  end;

  { Apply an XSL transformation to a DOM document.
    The document can come from an existing file (XSLSource),
    from a stream (XSLStream), or from memory (XSLText). }
  TXBBTransform = class(TXMLBuildingBlock)
  private
    FXSLSource: TFileName;
    FXSLStream: TStream;
    FXSLText: TStrings;
  protected
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent); overload; override;
    constructor Create(AOwner: TComponent; const XSLSource: TFileName);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const XSLStream: TStream);
      reintroduce; overload;
    constructor Create(AOwner: TComponent; const XSLText: TStrings);
      reintroduce; overload;
    destructor Destroy; override;
  published
    property Consumer;
    property XSLSource: TFileName read FXSLSource write FXSLSource;
    property XSLStream: TStream read FXSLStream write FXSLStream;
    property XSLText: TStrings read FXSLText write FXSLText;
  end;

  { Add a timestamp to a DOM. The timestamp appears as a separate element
    named from the TagName property (or 'timestamp' if that is blank)
    that appears first or last under the existing document's
    main element (depending on the InsertAtStart property.
    The Format property defines the appearance of the timestamp and uses the
    same notation as required by the FormatDateTime function.
    Format can be extended to generate multiple date parts beneath the
    timestamp element. Separate sub-elements with vertical bars ( | ),
    and sub-element names from formats with equals ( = ). For example,
    a Format of 'year=yyyy|month=MM|day=dd' creates the following structure:
    <timestamp><year>2002</year><month>03</month><day>07</day></timestamp>.
    Prefix a name with '@' to make it an attribute instead. For example,
    a Format of '@year=yyyy|@month=MM|@day=dd' creates the following structure:
    <timestamp year="2002" month="03" day="07"/>. }
  TXBBTimestamp = class(TXMLBuildingBlock)
  private
    FFormat: string;
    FInsertAtStart: Boolean;
    procedure SetFormat(const Value: string);
  protected
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent; const Format: string = '';
      const TagName: string = ''); reintroduce; overload;
  published
    property Consumer;
    property Format: string read FFormat write SetFormat;
    property InsertAtStart: Boolean read FInsertAtStart write FInsertAtStart;
    property TagName;
  end;

  { The node types }
  TXBBNodeType = (ntElement, ntAttribute, ntText, ntCDATA, ntEntityReference,
    ntEntity, ntProcessingInstr, ntComment, ntDocument, ntDocumentType,
    ntDocumentFragment, ntNotation);
  { The types of nodes to process }
  TXBBNodeTypes = set of TXBBNodeType;

  { An object wrapper of an interface for use in a tree. }
  TXBBNodePointer = class(TObject)
  public
    Node: IDOMNode;
    constructor Create(Node: IDOMNode);
  end;

  { Display a DOM within a tree view.
    The nodes as selected by the ShowNodes property are inserted into the
    attached tree view (TreeView property). Each node is assigned an
    image index corresponding to its node type. Assign an appropriate
    image list to the tree view to display these. You can use the
    DefaultNodeImages variable from this unit. }
  TXBBTreeView = class(TXMLBuildingBlock)
  private
    FShowNodes: TXBBNodeTypes;
    FTreeView: TTreeView;
    procedure SetTreeView(Value: TTreeView);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent); overload; override;
    constructor Create(AOwner: TComponent; const TreeView: TTreeView);
      reintroduce; overload;
    destructor Destroy; override;
    procedure ClearTree;
  published
    property Consumer;
    property ShowNodes: TXBBNodeTypes read FShowNodes write FShowNodes
      default [ntElement..ntNotation];
    property TreeView: TTreeView read FTreeView write SetTreeView;
  end;

  { Display a DOM within a memo.
    The XML text corresponding to the DOM is displayed in the attached
    memo (Memo property). }
  TXBBMemo = class(TXMLBuildingBlock)
  private
    FMemo: TMemo;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent; const Memo: TMemo = nil);
      reintroduce; overload;
  published
    property Consumer;
    property Memo: TMemo read FMemo write FMemo;
  end;

  { Display a DOM within a string grid.
    The values from the nodes immediately beneath the document element
    in the document provided are displayed in the attached string grid
    (StringGrid property).
    If SingleNode is True (the default), the nodes are shown in two columns.
    The first is the node's name, the second is the node's value.
    If SingleNode is False, the nodes are assumed to be multiple copies of
    the one type and so are shown in a two-dimensional format.
    Each row is a new child element of the document element,
    with each column being one of its children and value. }
  TXBBStringGrid = class(TXMLBuildingBlock)
  private
    FSingleNode: Boolean;
    FStringGrid: TStringGrid;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent); overload; override;
    constructor Create(AOwner: TComponent; const StringGrid: TStringGrid);
      reintroduce; overload;
  published
    property Consumer;
    property SingleNode: Boolean read FSingleNode write FSingleNode
      default True;
    property StringGrid: TStringGrid read FStringGrid write FStringGrid;
  end;

  { Display a DOM within a Web browser. }
  TXBBWebBrowser = class(TXMLBuildingBlock)
  private
    FWebBrowser: TWebBrowser;
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation);
      override;
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    constructor Create(AOwner: TComponent; const WebBrowser: TWebBrowser);
      reintroduce; overload;
  published
    property Consumer;
    property WebBrowser: TWebBrowser read FWebBrowser write FWebBrowser;
  end;

  { Create a new component from a DOM. }
  TXBBComponentCreate = class(TXMLBuildingBlock)
  private
    FComponent: TComponent;
  protected
    function ProcessDocument(const Document: IDOMDocument): IDOMDocument;
      override;
  public
    destructor Destroy; override;
    property Component: TComponent read FComponent;
  published
    property Consumer;
  end;

var
  { Default set of images for a tree view for XML node types.
    The image order follows the node types specified in the XMLDOM unit:
    from ELEMENT_NODE through to NOTATION_NODE. }
  DefaultNodeImages: TImageList;

{ Escape meta-characters in XML text. }
function EscapeText(Value: string): string;

implementation

{$R *.res}

{ Escape meta-characters in XML text. }
function EscapeText(Value: string): string;
begin
  Result := Value;
  Result := StringReplace(Result, '&', '&amp;', [rfReplaceAll]);
  Result := StringReplace(Result, '<', '&lt;', [rfReplaceAll]);
  Result := StringReplace(Result, '>', '&gt;', [rfReplaceAll]);
  Result := StringReplace(Result, '''', '&apos;', [rfReplaceAll]);
  Result := StringReplace(Result, '"', '&quot;', [rfReplaceAll]);
end;

{ TXMLBuildingBlock -----------------------------------------------------------}

{ Do nothing - implemented as needed in subclasses }
procedure TXMLBuildingBlock.CreateDocument;
begin
end;

{ Process the document according to this class, then pass it on to any consumer }
procedure TXMLBuildingBlock.DocumentReady(Document: IDOMDocument);
begin
  Document := ProcessDocument(Document);
  NotifyConsumer(Document);
end;

{ Trigger the tag create event }
procedure TXMLBuildingBlock.DoOnTagCreate(Element: IDOMElement);
begin
  if Assigned(OnTagCreate) then
    OnTagCreate(Self, Element);
end;

function TXMLBuildingBlock.GetConsumer: IXMLConsumer;
begin
  Result := FConsumer;
end;

{ Create a new document and document element }
function TXMLBuildingBlock.NewDocument(const TagName: string): IDOMDocument;
var
  DocElement: IDOMElement;
begin
  Result     := GetDOM().CreateDocument('', TagName, nil);
  DocElement := IDOMElement(Result.AppendChild(Result.CreateElement(TagName)));
  if TagName = Self.TagName then
    DoOnTagCreate(DocElement);
end;

{ Tidy up if attached components are deleted }
procedure TXMLBuildingBlock.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    { Need to check based on interfaces. }
    if Assigned(Consumer) and AComponent.IsImplementorOf(Consumer) then
      Consumer := nil;
  end;
end;

{ Pass the completed document onto any register consumer }
procedure TXMLBuildingBlock.NotifyConsumer(Document: IDOMDocument);
begin
  if Assigned(FConsumer) then
    FConsumer.DocumentReady(Document);
end;

{ Do nothing - overridden in subclasses }
function TXMLBuildingBlock.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;
begin
  Result := Document;
end;

{ Handle the assigned interface so that we get notifications about it }
procedure TXMLBuildingBlock.SetConsumer(const Value: IXMLConsumer);
begin
  ReferenceInterface(FConsumer, opRemove);
  FConsumer := Value;
  ReferenceInterface(FConsumer, opInsert);
end;

{ TXBBParser ------------------------------------------------------------------}

constructor TXBBParser.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FXMLText := TStringList.Create;
end;

{ Initialise and set the source as a file/URL }
constructor TXBBParser.Create(AOwner: TComponent; const XMLSource: TFileName);
begin
  Create(AOwner);
  Self.XMLSource := XMLSource;
end;

{ Initialise and set the source as a stream }
constructor TXBBParser.Create(AOwner: TComponent; const XMLStream: TStream);
begin
  Create(AOwner);
  Self.XMLStream := XMLStream;
end;

{ Initialise and set the source as a list of strings }
constructor TXBBParser.Create(AOwner: TComponent; const XMLText: TStrings);
begin
  Create(AOwner);
  Self.XMLText := XMLText;
end;

destructor TXBBParser.Destroy;
begin
  FXMLText.Free;
  inherited Destroy;
end;

{ Read the document in from the nominated source }
procedure TXBBParser.CreateDocument;
var
  Document: IDOMDocument;
begin
  if (XMLSource = '') and not Assigned(XMLStream) and (XMLText.Text = '') then
    raise EXBBException.Create('No source specified for XML document');
  Document := NewDocument('dummy');
  with Document as IDOMPersist do
    if XMLSource <> '' then
      Load(XMLSource)
    else if Assigned(XMLStream) then
      LoadFromStream(XMLStream)
    else
      LoadXML(XMLText.Text);
  DocumentReady(Document);
end;

{ TXBBSQL ---------------------------------------------------------------------}

constructor TXBBSQL.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FFieldFormat := xfText;
  FSQL         := TStringList.Create;
end;

{ Initialise and set the tag name to be used }
constructor TXBBSQL.Create(AOwner: TComponent; const TagName: string);
begin
  Create(AOwner);
  Self.TagName := TagName;
end;

{ Initialise and set the database and query to be used }
constructor TXBBSQL.Create(AOwner: TComponent; const DatabaseName: string;
  const SQL: TStrings; const TagName: string = '');
begin
  Create(AOwner, TagName);
  Self.DatabaseName := DatabaseName;
  Self.SQL          := SQL;
end;

{ Release resources }
destructor TXBBSQL.Destroy;
begin
  FSQL.Free;
  inherited Destroy;
end;

{ Run the query against the database and convert the results into XML }
procedure TXBBSQL.CreateDocument;
var
  Document: IDOMDocument;
  RecordElement, FieldElement: IDOMElement;
  Query: TQuery;
  Index: Integer;
  RecTagName, FieldName, FieldValue: string;
begin
  if (DatabaseName = '') or (SQL.Text = '') then
    raise EXBBException.Create('Missing database name or SQL');
  RecTagName := IfThen(RecordTagName <> '', RecordTagName, 'record');
  Document   := NewDocument(IfThen(TagName <> '', TagName, DatabaseName));
  Query      := TQuery.Create(nil);
  with Query do
    try
      DatabaseName := Self.DatabaseName;
      SQL          := Self.SQL;
      Open;
      for Index := 0 to FieldCount - 1 do
        if Fields[Index] is TMemoField then
          Fields[Index].OnGetText := GetText;
      while not Eof do
      begin
        { Create an element for each record }
        RecordElement := IDOMElement(Document.DocumentElement.AppendChild(
          Document.CreateElement(RecTagName)));
        if Assigned(OnRecordTagCreate) then
          OnRecordTagCreate(Self, RecordElement, Query);
        for Index := 0 to FieldCount - 1 do
        begin
          FieldName  := Fields[Index].DisplayName;
          FieldValue := EscapeText(Fields[Index].DisplayText);
          case FieldFormat of
            xfText:
              { And then a sub-element for each field }
              begin
                FieldElement := IDOMElement(RecordElement.AppendChild(
                  Document.CreateElement(FieldName)));
                FieldElement.AppendChild(Document.CreateTextNode(FieldValue));
              end;
            xfElement:
              { Add field values as attributes on separate elements }
              begin
                FieldElement := IDOMElement(RecordElement.AppendChild(
                  Document.CreateElement(FieldName)));
                FieldElement.setAttribute('value', FieldValue);
              end;
            xfAttributeOnly:
              { Add field values as attributes on the record element }
              RecordElement.setAttribute(FieldName, FieldValue);
          end;
        end;
        Next;
      end;
      Close;
      DocumentReady(Document);
    finally
      Free;
    end;
end;

{ Retrieve the contents of a memo field }
procedure TXBBSQL.GetText(Sender: TField; var Text: String;
  DisplayText: Boolean);
begin
  Text := TMemoField(Sender).AsString
end;

{ TXBBTextFile ----------------------------------------------------------------}

{ Initialise and optionally set the filename and tag name }
constructor TXBBTextFile.Create(AOwner: TComponent;
  const FileName: TFileName = ''; const TagName: string = '');
begin
  inherited Create(AOwner);
  Self.FileName := FileName;
  Self.TagName  := TagName;
end;

{ Read a text file and wrap it in an element }
procedure TXBBTextFile.CreateDocument;
var
  Document: IDOMDocument;
  Text: TStringList;
begin
  if FileName = '' then
    raise EXBBException.Create('Missing filename');
  Document := NewDocument(IfThen(TagName <> '', TagName, 'file'));
  Text     := TStringList.Create;
  try
    Text.LoadFromFile(FileName);
    Document.DocumentElement.SetAttribute('filename', FileName);
    if AsCDATA then
      Document.DocumentElement.AppendChild(Document.CreateCDATASection(Text.Text))
    else
      Document.DocumentElement.AppendChild(
        Document.CreateTextNode(EscapeText(Text.Text)));
  finally
    Text.Free;
  end;
  DocumentReady(Document);
end;

{ TXBBComponent ---------------------------------------------------------------}

{ Initialise and optionally set the tag name and component to wrap }
constructor TXBBComponent.Create(AOwner: TComponent; const TagName: string = '';
  const Component: TComponent = nil);
begin
  inherited Create(AOwner);
  Self.TagName   := TagName;
  Self.Component := Component;
end;

{ Serialise a component into an XML document }
procedure TXBBComponent.CreateDocument;
var
  Document: IDOMDocument;
  Element: IDOMElement;
  Text: TStringList;
  Index: Integer;
  Line: string;

  { Serialise a component into a string (like the DFM) }
  function ComponentToString(Component: TComponent): string;
  var
    MemStream: TMemoryStream;
    StrStream: TStringStream;
  begin
    MemStream := TMemoryStream.Create;
    try
      StrStream := TStringStream.Create('');
      try
        MemStream.WriteComponent(Component);
        MemStream.Seek(0, soFromBeginning);
        ObjectBinaryToText(MemStream, StrStream);
        StrStream.Seek(0, soFromBeginning);
        Result := StrStream.DataString;
      finally
        StrStream.Free;
      end;
    finally
      MemStream.Free
    end;
  end;

  { Create an object element }
  procedure StartObject(NameAndType: string);
  var
    Index: Integer;
  begin
    Element :=
      IDOMElement(Element.AppendChild(Document.CreateElement('object')));
    Index   := Pos(':', NameAndType);
    Element.SetAttribute('name', Trim(Copy(NameAndType, 1, Index - 1)));
    Element.SetAttribute('type',
      Trim(Copy(NameAndType, Index + 1, Length(NameAndType))));
  end;

  { Create a collection item element }
  procedure StartItem;
  begin
    Element := IDOMElement(Element.AppendChild(Document.CreateElement('item')));
  end;

  { Finish an object or item element and move back up the DOM tree }
  procedure EndObjectOrItem;
  begin
    Element := IDOMElement(Element.ParentNode);
  end;

  { Finish a collection property element and move back up the DOM tree }
  procedure EndCollection;
  begin
    Element := IDOMElement(Element.ParentNode.ParentNode);
  end;

  { Add nodes for a normal component property }
  procedure AddProperty(NameAndValue: string);
  var
    Index: Integer;
    Value: string;
  begin
    with IDOMElement(Element.AppendChild(Document.CreateElement('property'))) do
    begin
      Index := Pos('=', NameAndValue);
      Value := Trim(Copy(NameAndValue, Index + 1, Length(NameAndValue)));
      SetAttribute('name', Trim(Copy(NameAndValue, 1, Index - 1)));
      if Value <> '' then
        SetAttribute('value', Value);
    end;
  end;

  { Add nodes for a component property with binary data }
  procedure AddBinaryProperty(NameAndValue: string);
  begin
    repeat
      Inc(Index);
      NameAndValue := NameAndValue + Trim(Text[Index]);
    until Pos('}', Text[Index]) > 0;
    AddProperty(NameAndValue);
  end;

  { Add nodes for a component property with a list of string values }
  procedure AddListProperty(NameAndValue: string);
  begin
    repeat
      Inc(Index);
      NameAndValue := NameAndValue + Trim(Text[Index]) + '|';
    until Text[Index][Length(Text[Index])] = ')';
    AddProperty(Copy(NameAndValue, 1, Length(NameAndValue) - 1));
  end;

  { Add nodes for a component property which is a collection }
  procedure AddCollectionProperty(NameAndValue: string);
  var
    Index: Integer;
  begin
    Element :=
      IDOMElement(Element.AppendChild(Document.CreateElement('property')));
    Index   := Pos('=', NameAndValue);
    Element.SetAttribute('name', Trim(Copy(NameAndValue, 1, Index - 1)));
  end;

  { Add nodes for a component property with a long string }
  procedure AddMultilineProperty(NameAndValue: string);
  begin
    NameAndValue := NameAndValue + ' |';
    repeat
      Inc(Index);
      NameAndValue := NameAndValue + Trim(Text[Index]) + '|';
    until Text[Index][Length(Text[Index])] <> '+';
    AddProperty(Copy(NameAndValue, 1, Length(NameAndValue) - 1));
  end;

begin
  if not Assigned(Component) then
    raise EXBBException.Create('Missing component');
  Document := NewDocument(IfThen(TagName <> '', TagName, 'component'));
  Element  := Document.DocumentElement;
  Text     := TStringList.Create;
  try
    { Serialise component to a list of strings }
    Text.Text := ComponentToString(Component);
    Index := 0;
    while Index < Text.Count do
    begin
      { Then process these according to type }
      Line := Trim(Text[Index]);
      if Copy(Line, 1, 6) = 'object' then
        StartObject(Copy(Line, 8, Length(Line)))
      else if Copy(Line, 1, 4) = 'item' then
        StartItem
      else if Line = 'end'  then
        EndObjectOrItem
      else if Line = 'end>'  then
        EndCollection
      else if Pos('= {', Line) = Length(Line) - 2 then
        AddBinaryProperty(Line)
      else if Pos('= (', Line) = Length(Line) - 2 then
        AddListProperty(Line)
      else if Pos('= <', Line) = Length(Line) - 2 then
        AddCollectionProperty(Line)
      else if Pos('=', Line) = Length(Line) then
        AddMultilineProperty(Line)
      else
        AddProperty(Line);
      Inc(Index);
    end;
  finally
    Text.Free;
  end;
  DocumentReady(Document);
end;

{ TXBBWriter ------------------------------------------------------------------}

{ Initialise and set the destination filename/URL }
constructor TXBBWriter.Create(AOwner: TComponent; const FileName: TFileName);
begin
  inherited Create(AOwner);
  Self.Filename := Filename;
end;

{ Initialise and set the destination stream }
constructor TXBBWriter.Create(AOwner: TComponent; const Stream: TStream);
begin
  inherited Create(AOwner);
  Self.Stream := Stream;
end;

{ Write the document out to the specified destination }
function TXBBWriter.ProcessDocument(const Document: IDOMDocument): IDOMDocument;
begin
  if (Filename = '') and not Assigned(Stream) then
    raise EXBBException.Create('No filename or stream specified');
  with Document as IDOMPersist do
    if Assigned(Stream) then
      SaveToStream(Stream)
    else
      Save(Filename);
  Result := Document;
end;

{ TXBBMerge -------------------------------------------------------------------}

{ Initialise and optionally set properties }
constructor TXBBMerge.Create(AOwner: TComponent;
  const NumDocuments: Integer = 2; const TagName: string = '');
begin
  inherited Create(AOwner);
  Self.NumDocuments := NumDocuments;
  Self.TagName      := TagName;
end;

{ Add the given document into the merged one, and pass on when finished }
procedure TXBBMerge.DocumentReady(Document: IDOMDocument);
begin
  if not Assigned(FMergedDocument) then
    FMergedDocument := NewDocument(IfThen(TagName <> '', TagName, 'merge'));
  FMergedDocument.DocumentElement.AppendChild(Document.DocumentElement);
  Dec(FCountDown);
  if FCountDown = 0 then
  begin
    { Document is complete }
    NotifyConsumer(FMergedDocument);
    FCountDown := NumDocuments;
  end;
end;

procedure TXBBMerge.SetNumDocuments(Value: Integer);
begin
  FNumDocuments   := Value;
  FCountDown      := Value;
  FMergedDocument := nil;
end;

{ TXBBConsumerCollectionItem --------------------------------------------------}

{ Return value for display in property editor }
function TXBBConsumerCollectionItem.GetDisplayName: string;
begin
  Result := IfThen(Name = '', '<Unknown>', Name);
end;

procedure TXBBConsumerCollectionItem.SetConsumer(Value: IXMLConsumer);
begin
  FConsumer := Value;
  Changed(False);
end;

procedure TXBBConsumerCollectionItem.SetName(Value: string);
begin
  FName := Value;
  Changed(False);
end;

{ TXBBConsumerCollection ------------------------------------------------------}

{ Initialise and set owner }
constructor TXBBConsumerCollection.Create(Owner: TPersistent);
begin
  inherited Create(TXBBConsumerCollectionItem);
  FOwner := Owner;
end;

function TXBBConsumerCollection.Add: TXBBConsumerCollectionItem;
begin
  Result := TXBBConsumerCollectionItem(inherited Add);
end;

function TXBBConsumerCollection.FindItemID(ID: Integer):
  TXBBConsumerCollectionItem;
begin
  Result := TXBBConsumerCollectionItem(inherited FindItemID(ID));
end;

function TXBBConsumerCollection.GetItem(Index: Integer):
  TXBBConsumerCollectionItem;
begin
  Result := TXBBConsumerCollectionItem(inherited GetItem(Index));
end;

function TXBBConsumerCollection.GetOwner: TPersistent;
begin
  Result := FOwner;
end;

function TXBBConsumerCollection.Insert(Index: Integer):
  TXBBConsumerCollectionItem;
begin
  Result := TXBBConsumerCollectionItem(inherited Insert(Index));
end;

procedure TXBBConsumerCollection.SetItem(Index: Integer;
  Value: TXBBConsumerCollectionItem);
begin
  inherited SetItem(Index, Value);
end;

{ TXBBFork --------------------------------------------------------------------}

constructor TXBBFork.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FConsumers := TXBBConsumerCollection.Create(Self);
end;

destructor TXBBFork.Destroy;
begin
  FConsumers.Free;
  inherited Destroy;
end;

{ Copy the document and pass it on to each consumer }
procedure TXBBFork.DocumentReady(Document: IDOMDocument);
var
  Index: Integer;
begin
  for Index := 0 to FConsumers.Count - 1 do
    if Assigned(FConsumers.Items[Index].Consumer) then
      FConsumers.Items[Index].Consumer.
        DocumentReady(Document.CloneNode(True) as IDOMDocument);
end;

{ TXBBTransform ---------------------------------------------------------------}

constructor TXBBTransform.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FXSLText := TStringList.Create;
end;

{ Initialise and set the source as a file/URL }
constructor TXBBTransform.Create(AOwner: TComponent; const XSLSource: TFileName);
begin
  Create(AOwner);
  Self.XSLSource := XSLSource;
end;

{ Initialise and set the source as a stream }
constructor TXBBTransform.Create(AOwner: TComponent; const XSLStream: TStream);
begin
  Create(AOwner);
  Self.XSLStream := XSLStream;
end;

{ Initialise and set the source as a list of strings }
constructor TXBBTransform.Create(AOwner: TComponent; const XSLText: TStrings);
begin
  Create(AOwner);
  Self.XSLText := XSLText;
end;

destructor TXBBTransform.Destroy;
begin
  FXSLText.Free;
  inherited Destroy;
end;

{ Apply the transformation specified earlier to the document }
function TXBBTransform.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;
var
  XSLDocument: IDOMDocument;
  OK: Boolean;
begin
  if (XSLSource = '') and not Assigned(XSLStream) and (XSLText.Text = '') then
    raise EXBBException.Create('No source specified for XSL document');
  XSLDocument := NewDocument('xslt');
  with XSLDocument as IDOMPersist do
    if XSLSource <> '' then
      OK := Load(XSLSource)
    else if Assigned(XSLStream) then
      OK := LoadFromStream(XSLStream)
    else
      OK := LoadXML(XSLText.Text);
  if not OK then
    raise EXBBException.Create((XSLDocument as IDOMParseError).reason);
  Result := NewDocument('out');
  (Document as IDOMNodeEx).TransformNode(XSLDocument, Result);
end;

{ TXBBTimestamp ---------------------------------------------------------------}

constructor TXBBTimestamp.Create(AOwner: TComponent;
  const Format: string = ''; const TagName: string = '');
begin
  inherited Create(AOwner);
  Self.Format  := Format;
  Self.TagName := TagName;
end;

{ Add a timestamp element (or subtree) to the document }
function TXBBTimestamp.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;
var
  Element: IDOMElement;
  DateTime: TDateTime;

  { Format consists of several fields to generate multiple date parts beneath
    the timestamp element. Separate sub-elements with vertical bars ( | ),
    and sub-element names from formats with equals ( = ). For example,
    a Format of 'year=yyyy|month=MM|day=dd' creates the following structure:
    <timestamp><year>2002</year><month>03</month><day>07</day></timestamp>.
    Prefix a name with '@' to make it an attribute instead. For example,
    a Format of '@year=yyyy|@month=MM|@day=dd' creates the following structure:
    <timestamp year="2002" month="03" day="07"/>. }
  procedure AddSubFormats(MainElement: IDOMElement);
  var
    Index: Integer;
    Name, SubFormat, WorkFormat: string;
  begin
    WorkFormat := Format;
    repeat
      Index := Pos('=', WorkFormat);
      if Index = 0 then
        Exit;
      Name := Copy(WorkFormat, 1, Index - 1);
      Delete(WorkFormat, 1, Index);
      Index := Pos('|', WorkFormat);
      if Index = 0 then
        Index := Length(WorkFormat) + 1;
      SubFormat := Copy(WorkFormat, 1, Index - 1);
      Delete(WorkFormat, 1, Index);
      if Name[1] = '@' then
        MainElement.SetAttribute(Copy(Name, 2, Length(Name)),
          FormatDateTime(SubFormat, DateTime))
      else
        MainElement.AppendChild(Document.CreateElement(Name)).
          AppendChild(Document.CreateTextNode(
          FormatDateTime(SubFormat, DateTime)));
    until WorkFormat = '';
  end;

begin
  DateTime := Now;
  Element  := Document.CreateElement(IfThen(TagName <> '', TagName, 'timestamp'));
  if Pos('=', Format) = 0 then
    Element.AppendChild(Document.CreateTextNode(FormatDateTime(Format, DateTime)))
  else
    AddSubFormats(Element);
  if InsertAtStart then
    Document.DocumentElement.InsertBefore(
      Element, Document.DocumentElement.FirstChild)
  else
    Document.DocumentElement.AppendChild(Element);
  Result := Document;
end;

procedure TXBBTimestamp.SetFormat(const Value: string);
begin
  FFormat := IfThen(Value <> '', Value, ShortDateFormat);
end;

{ TXBBNodePointer -------------------------------------------------------------}

constructor TXBBNodePointer.Create(Node: IDOMNode);
begin
  inherited Create;
  Self.Node := Node;
end;

{ TXBBTreeView ----------------------------------------------------------------}

constructor TXBBTreeView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FShowNodes := [ntElement..ntNotation];
end;

{ Initialise and set tree view to update }
constructor TXBBTreeView.Create(AOwner: TComponent; const TreeView: TTreeView);
begin
  Create(AOwner);
  FTreeView := TreeView;
end;

{ Free up resources }
destructor TXBBTreeView.Destroy;
begin
  ClearTree;
  inherited Destroy;
end;

{ Free up objects in the tree }
procedure TXBBTreeView.ClearTree;
var
  Index: Integer;
begin
  if not Assigned(FTreeView) or (csDestroying in FTreeView.ComponentState) then
    Exit;
  for Index := 0 to FTreeView.Items.Count - 1 do
    if Assigned(FTreeView.Items[Index].Data) then
      TObject(FTreeView.Items[Index].Data).Free;
  FTreeView.Items.Clear;
end;

{ Tidy up if attached components are deleted }
procedure TXBBTreeView.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if Assigned(TreeView) and (TreeView = AComponent) then
      TreeView := nil;
  end;
end;

{ Fill the tree view with nodes corresponding to the document's nodes }
function TXBBTreeView.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;

  { Recursively add nodes to the tree while stepping through the DOM structure }
  procedure PopulateTree(Node: IDOMNode; Parent: TTreeNode);
  var
    DisplayName: string;
    Index: Integer;
    NewNode: TTreeNode;
  begin
    if not (TXBBNodeType(Node.NodeType - 1) in ShowNodes) then
      Exit;
    case Node.NodeType of
      DOCUMENT_NODE:       DisplayName := 'Document';
      TEXT_NODE,
      CDATA_SECTION_NODE,
      COMMENT_NODE:        DisplayName := Node.NodeValue;
      else                 DisplayName := Node.NodeName;
    end;
    NewNode := TreeView.Items.AddChildObject(Parent, DisplayName,
      TXBBNodePointer.Create(Node));
    { Select images based on node type }
    NewNode.ImageIndex    := Node.NodeType - 1;
    NewNode.SelectedIndex := NewNode.ImageIndex;
    if Assigned(Node.Attributes) then
      for Index := 0 to Node.Attributes.Length - 1 do
        PopulateTree(Node.Attributes.Item[Index], NewNode);
    for Index := 0 to Node.ChildNodes.Length - 1 do
      PopulateTree(Node.ChildNodes.Item[Index], NewNode);
  end;

begin
  if not Assigned(TreeView) then
    Exit;
  TreeView.Items.BeginUpdate;
  ClearTree;
  PopulateTree(Document, nil);
  TreeView.Items.EndUpdate;
  Result := Document;
end;

procedure TXBBTreeView.SetTreeView(Value: TTreeView);
begin
  ClearTree;
  FTreeView := Value;
end;

{ TXBBMemo --------------------------------------------------------------------}

{ Initialise and optionally set the memo to fill }
constructor TXBBMemo.Create(AOwner: TComponent; const Memo: TMemo = nil);
begin
  inherited Create(AOwner);
  FMemo := Memo;
end;

{ Tidy up if attached components are deleted }
procedure TXBBMemo.Notification(AComponent: TComponent; Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if Assigned(Memo) and (Memo = AComponent) then
      Memo := nil;
  end;
end;

{ Copy the DOM as text into the memo }
function TXBBMemo.ProcessDocument(const Document: IDOMDocument): IDOMDocument;
var
  Stream: TMemoryStream;
begin
  if not Assigned(Memo) then
    Exit;
  Memo.Lines.Clear;
  Stream := TMemoryStream.Create;
  try
    (Document as IDOMPersist).SaveToStream(Stream);
    Stream.Position := 0;
    Memo.Lines.LoadFromStream(Stream);
  finally
    Stream.Free;
  end;
  Result := Document;
end;

{ TXBBStringGrid --------------------------------------------------------------}

constructor TXBBStringGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSingleNode := True;
end;

{ Initialise and set the string grid to fill }
constructor TXBBStringGrid.Create(AOwner: TComponent;
  const StringGrid: TStringGrid);
begin
  Create(AOwner);
  FStringGrid := StringGrid;
end;

{ Tidy up if attached components are deleted }
procedure TXBBStringGrid.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if Assigned(StringGrid) and (StringGrid = AComponent) then
      StringGrid := nil;
  end;
end;

{ Fill a string grid with values from the DOM }
function TXBBStringGrid.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;

  { Compile the text content of a node - down to one level below }
  function GetText(Node: IDOMNode): string;
  var
    Index: Integer;
  begin
    Result := Node.NodeValue;
    for Index := 0 to Node.ChildNodes.Length - 1 do
      Result := Result + Node.ChildNodes.Item[Index].NodeValue + ' ';
  end;

  { Present the DOM, i.e. the nodes beneath the document element,
    in two columns: one for the node name, one for its value }
  procedure DoSingleNode;
  var
    Element: IDOMElement;
    Node: IDOMNode;
    Index: Integer;
  begin
    with StringGrid do
    begin
      Element     := Document.DocumentElement;
      ColCount    := 2;
      RowCount    := Element.ChildNodes.Length + 1;
      FixedCols   := 0;
      FixedRows   := 1;
      Cells[0, 0] := 'Element';
      Cells[1, 0] := 'Value';
      for Index := 0 to Element.ChildNodes.Length - 1 do
      begin
        Node                := Element.ChildNodes.Item[Index];
        Cells[0, Index + 1] := Node.NodeName;
        Cells[1, Index + 1] := GetText(Node);
      end;
    end;
  end;

  { Present the DOM, i.e. the nodes beneath the document element,
    in multiple columns. This assumes that the document element
    contains multiple instances of the same type of element.
    Each row is then one of these elements, and each column is
    one of its sub-elements. }
  procedure DoMultipleNodes;
  var
    Element: IDOMElement;
    Node, Node2: IDOMNode;
    Index, Index2, Column: Integer;
    Text: string;
  begin
    with StringGrid do
    begin
      Element      := Document.DocumentElement;
      ColCount     := 2;
      RowCount     := Max(2, Element.ChildNodes.Length + 1);
      FixedCols    := 1;
      FixedRows    := 1;
      Cells[0, 0]  := '#';
      Cells[1, 0]  := '#text';
      Cells[0, 1]  := '';
      Cells[1, 1]  := '';
      ColWidths[0] := 20;
      ColWidths[1] := 20;
      for Index := 0 to Element.ChildNodes.Length - 1 do
      begin
        Node := Element.ChildNodes.Item[Index];
        Rows[Index + 1].Text := '';
        Cells[0, Index + 1]  := IntToStr(Index + 1);
        for Index2 := 0 to Node.ChildNodes.Length - 1 do
        begin
          Node2 := Node.ChildNodes.Item[Index2];
          Text  := GetText(Node2);
          for Column := 1 to ColCount - 1 do
            if Node2.NodeName = Cells[Column, 0] then
              Break;
          ColCount                 := Max(ColCount, Column + 1);
          Cells[Column, 0]         := Node2.NodeName;
          Cells[Column, Index + 1] := Text;
          ColWidths[Column]        :=
            Max(20, Min(ColWidths[Column], Canvas.TextWidth(Text)));
        end;
      end;
    end;
  end;

begin
  if not Assigned(StringGrid) then
    Exit;
  if SingleNode then
    DoSingleNode
  else
    DoMultipleNodes;
  Result := Document;
end;

{ TXBBWebBrowser --------------------------------------------------------------}

{ Initialise and set the Web browser to write to }
constructor TXBBWebBrowser.Create(AOwner: TComponent;
  const WebBrowser: TWebBrowser);
begin
  Create(AOwner);
  FWebBrowser := WebBrowser;
end;

{ Tidy up if attached components are deleted }
procedure TXBBWebBrowser.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if Operation = opRemove then
  begin
    if Assigned(WebBrowser) and (WebBrowser = AComponent) then
      WebBrowser := nil;
  end;
end;

{ Copy the DOM contents into the Web browser }
function TXBBWebBrowser.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;
var
  TempStream: TMemoryStream;
  StreamAdapter: TStreamAdapter;
begin
  if not Assigned(WebBrowser) then
    Exit;
  TempStream := TMemoryStream.Create;
  try
    (Document as IDOMPersist).saveToStream(TempStream);
    TempStream.Position := 0;
    StreamAdapter := TStreamAdapter.Create(TempStream);
    if WebBrowser.Document = nil then
      WebBrowser.Navigate('about:blank');
    (WebBrowser.Document as IPersistStreamInit).Load(StreamAdapter);
  finally
    TempStream.Free;
  end;
  Result := Document;
end;

{ TXBBComponentCreate ---------------------------------------------------------}

destructor TXBBComponentCreate.Destroy;
begin
  if Assigned(FComponent) and not (csDestroying in FComponent.ComponentState) then
    FComponent.Free;
  inherited Destroy;
end;

{ Create a new component from the DOM contents }
function TXBBComponentCreate.ProcessDocument(const Document: IDOMDocument):
  IDOMDocument;
var
  Stream: TStringStream;

  { Reconstruct DFM contents from DOM nodes }
  procedure RebuildDFM(Element: IDOMElement; Stream: TStringStream;
    Indent: string);
  var
    Index: Integer;
    Name, Value: string;
    FirstCh: Char;

    { Reconstruct a binary value - segment into 64 character blocks }
    procedure RebuildBinary(Indent: string);
    var
      Index: Integer;
    begin
      Stream.WriteString(Indent + Name + ' = ' + FirstCh + #13#10);
      Indent := Indent + '  ';
      Index  := 2;
      repeat
        Stream.WriteString(Indent + Copy(Value, Index, 64) + #13#10);
        Inc(Index, 64);
      until Index > Length(Value);
    end;

    { Reconstruct a multi-line value - separate at vertical bars }
    procedure RebuildList(Indent, Header: string);
    var
      Start, Finish: Integer;
    begin
      Stream.WriteString(Indent + Name + ' =' + Header + #13#10);
      Indent := Indent + '  ';
      Start  := 2;
      Finish := 1;
      repeat
        repeat
          if Value[Finish] = '''' then
            { Skip over string values }
            repeat
              Inc(Finish);
            until Value[Finish] = '''';
          Inc(Finish);
        until (Finish > Length(Value)) or (Value[Finish] = '|');
        Stream.WriteString(Indent + Copy(Value, Start, Finish - Start) + #13#10);
        Start := Finish + 1;
      until Start > Length(Value);
    end;

  begin
    Name  := Element.GetAttribute('name');
    Value := Element.GetAttribute('value');
    if Element.NodeName = 'object' then
      Stream.WriteString(Indent + 'object ' + Name + ': ' +
        Element.GetAttribute('type') + #13#10)
    else if Element.NodeName = 'property' then
    begin
      if Value = '' then
        Value := '<';
      FirstCh := Value[1];
      case FirstCh of
        '<': Stream.WriteString(Indent + Name + ' = ' + FirstCh + #13#10);
        '{': RebuildBinary(Indent);
        '(': RebuildList(Indent, ' (');
        '|': RebuildList(Indent, '');
        else Stream.WriteString(Indent + Name + ' = ' + Value + #13#10);
      end;
    end
    else if Element.NodeName = 'item' then
    begin
      if Element.ChildNodes.Length > 0 then
        Stream.WriteString(Indent + 'item'#13#10);
    end;
    for Index := 0 to Element.ChildNodes.Length - 1 do
      if Element.ChildNodes.Item[Index].NodeType = ELEMENT_NODE then
        RebuildDFM(Element.ChildNodes.Item[Index] as IDOMElement,
          Stream, Indent + '  ');
    if Element.NodeName = 'object' then
      Stream.WriteString(Indent + 'end'#13#10)
    else if (Element.NodeName = 'property') and (FirstCh = '<') then
      Stream.WriteString(Indent + '>'#13#10)
    else if Element.NodeName = 'item' then
    begin
      if Element.ChildNodes.Length > 0 then
        Stream.WriteString(Indent + 'end'#13#10);
    end;
  end;

  { De-serialise from text to component }
  function StreamToComponent(Stream: TStream): TComponent;
  var
    MemStream: TMemoryStream;
  begin
    MemStream := TMemoryStream.Create;
    try
      Stream.Seek(0, soFromBeginning);
      ObjectTextToBinary(Stream, MemStream);
      MemStream.Seek(0, soFromBeginning);
      Result := MemStream.ReadComponent(nil);
    finally
      MemStream.Free;
    end;
  end;

begin
  FreeAndNil(FComponent);
  if Document.DocumentElement.NodeName <> 'component' then
    raise EXBBException.Create('XML document does not contain a component description');
  Stream := TStringStream.Create('');
  try
    RebuildDFM(Document.DocumentElement, Stream, '');
    FComponent := StreamToComponent(Stream);
  finally
    Stream.Free;
  end;
  Result := Document;
end;

initialization
  { Set up the default set of images for nodes in a tree view }
  DefaultNodeImages := TImageList.CreateSize(16, 16);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'DOCUMENT', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'ELEMENT_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'ATTRIBUTE_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'TEXT_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'CDATA_SECTION_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'ENTITY_REFERENCE_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'ENTITY_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'PROCESSING_INSTRUCTION_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'COMMENT_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'DOCUMENT_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'DOCUMENT_TYPE_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'DOCUMENT_FRAGMENT_NODE', clFuchsia);
  DefaultNodeImages.ResourceLoad(rtBitmap, 'NOTATION_NODE', clFuchsia);
finalization
  DefaultNodeImages.Free;
end.
